(function(){
  const nav = document.querySelector('.site-nav');
  const toggle = document.querySelector('.nav-toggle');
  const prefersReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)').matches;

  if (nav) {
    let ticking = false;
    const updateNav = () => {
      nav.classList.toggle('scrolled', window.scrollY > 10);
      ticking = false;
    };
    const onScroll = () => {
      if (!ticking) {
        ticking = true;
        requestAnimationFrame(updateNav);
      }
    };
    window.addEventListener('scroll', onScroll, { passive: true });
    updateNav();
  }

  const setNavState = (open) => {
    if (!nav || !toggle) return;
    nav.classList.toggle('open', open);
    toggle.setAttribute('aria-expanded', String(open));
    document.body.classList.toggle('nav-open', open);
  };

  if (toggle && nav) {
    toggle.addEventListener('click', () => {
      const expanded = toggle.getAttribute('aria-expanded') === 'true';
      setNavState(!expanded);
    });
  }

  const dropdowns = document.querySelectorAll('.nav-dropdown');
  const closeDropdown = (dropdown) => {
    const toggleEl = dropdown.querySelector('.nav-dropdown-toggle');
    dropdown.dataset.open = 'false';
    if (toggleEl) toggleEl.setAttribute('aria-expanded', 'false');
  };

  const closeNavMenu = () => {
    setNavState(false);
    dropdowns.forEach((dropdown) => closeDropdown(dropdown));
  };

  dropdowns.forEach((dropdown) => {
    const toggleEl = dropdown.querySelector('.nav-dropdown-toggle');
    if (!toggleEl) return;
    toggleEl.addEventListener('click', (event) => {
      event.preventDefault();
      const isOpen = dropdown.dataset.open === 'true';
      dropdowns.forEach((item) => {
        if (item !== dropdown) closeDropdown(item);
      });
      dropdown.dataset.open = String(!isOpen);
      toggleEl.setAttribute('aria-expanded', String(!isOpen));
    });
  });

  document.addEventListener('click', (event) => {
    dropdowns.forEach((dropdown) => {
      if (!dropdown.contains(event.target)) {
        closeDropdown(dropdown);
      }
    });
  });

  document.addEventListener('keydown', (event) => {
    if (event.key === 'Escape') {
      dropdowns.forEach((dropdown) => closeDropdown(dropdown));
      closeNavMenu();
    }
  });

  document.querySelectorAll('.nav-menu a').forEach((link) => {
    link.addEventListener('click', closeNavMenu);
  });

  document.addEventListener('click', (event) => {
    if (!nav || !nav.classList.contains('open')) return;
    if (!nav.contains(event.target)) closeNavMenu();
  });

  const revealEls = document.querySelectorAll('[data-reveal]');
  if (revealEls.length && !prefersReducedMotion) {
    const observer = new IntersectionObserver((entries) => {
      entries.forEach((entry) => {
        if (entry.isIntersecting) {
          entry.target.classList.add('visible');
          observer.unobserve(entry.target);
        }
      });
    }, { threshold: 0.2, rootMargin: '0px 0px -40px 0px' });
    revealEls.forEach((el) => {
      el.classList.add('reveal');
      observer.observe(el);
    });
  } else {
    revealEls.forEach((el) => el.classList.add('visible'));
  }

  const counters = document.querySelectorAll('[data-counter]');
  const animateCounter = (el) => {
    if (el.dataset.counterStarted) return;
    el.dataset.counterStarted = 'true';
    const target = Number(el.dataset.counter) || 0;
    const duration = 1600;
    const start = performance.now();
    const initial = Number(el.textContent.replace(/[^0-9.]/g, '')) || 0;

    const tick = (now) => {
      const progress = Math.min((now - start) / duration, 1);
      const value = initial + (target - initial) * progress;
      el.textContent = target % 1 === 0 ? Math.round(value).toLocaleString() : value.toFixed(1);
      if (progress < 1) requestAnimationFrame(tick);
    };
    requestAnimationFrame(tick);
  };

  if (counters.length && !prefersReducedMotion) {
    const counterObserver = new IntersectionObserver((entries) => {
      entries.forEach((entry) => {
        if (entry.isIntersecting) {
          animateCounter(entry.target);
          counterObserver.unobserve(entry.target);
        }
      });
    }, { threshold: 0.6 });
    counters.forEach((el) => counterObserver.observe(el));
  } else {
    counters.forEach((el) => animateCounter(el));
  }

  const parallaxEls = document.querySelectorAll('[data-parallax]');
  if (parallaxEls.length && !prefersReducedMotion) {
    const applyParallax = () => {
      const viewportMid = window.innerHeight / 2;
      parallaxEls.forEach((el) => {
        const rect = el.getBoundingClientRect();
        const offset = rect.top + rect.height / 2 - viewportMid;
        const speed = parseFloat(el.dataset.parallax) || 0.1;
        const translate = -offset * speed;
        el.style.setProperty('--parallax-offset', `${translate}px`);
      });
    };
    const handleScroll = () => requestAnimationFrame(applyParallax);
    window.addEventListener('scroll', handleScroll, { passive: true });
    window.addEventListener('resize', handleScroll);
    applyParallax();
  }

  const logoTrack = document.querySelector('.logo-track');
  if (logoTrack && logoTrack.children.length) {
    const clones = Array.from(logoTrack.children);
    clones.forEach((node) => {
      const clone = node.cloneNode(true);
      clone.setAttribute('aria-hidden', 'true');
      logoTrack.appendChild(clone);
    });
  }

  const activePage = document.body.dataset.page;
  const navItems = document.querySelectorAll('.site-nav [data-page]');
  navItems.forEach((item) => {
    if (activePage && item.dataset.page === activePage) {
      item.setAttribute('aria-current', 'page');
    } else {
      item.removeAttribute('aria-current');
    }
  });
  const toolPages = [
    'google-map-scraper',
    'email-finder',
    'email-sender',
    'email-verifier',
    'social-link-extractors',
    'yellow-pages-scraper',
    'yelp-scraper',
  ];
  if (activePage && toolPages.includes(activePage)) {
    const toolsToggle = document.querySelector('.nav-dropdown-toggle[data-page="tools"]');
    if (toolsToggle) toolsToggle.setAttribute('aria-current', 'page');
  }

  const footerYearTargets = document.querySelectorAll('[data-year], #year');
  if (footerYearTargets.length) {
    const yearValue = new Date().getFullYear();
    footerYearTargets.forEach((node) => {
      node.textContent = yearValue;
    });
  }
})();
